﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Management;
using System.Windows.Forms;

namespace RoboclawClassLib
{
    class USBDeviceInfo
    {
        public USBDeviceInfo(string name, string deviceID, string pnpDeviceID, string description)
        {
            this.Name = name;
            this.DeviceID = deviceID;
            this.PnpDeviceID = pnpDeviceID;
            this.Description = description;
        }
        public string Name { get; private set; }
        public string DeviceID { get; private set; }
        public string PnpDeviceID { get; private set; }
        public string Description { get; private set; }
    }

    public class Roboclaw
    {
        private System.IO.Ports.SerialPort serialPort;

        private System.Windows.Forms.TextBox debug=null;
        private Boolean updateread;

        static List<USBDeviceInfo> GetUSBDevices()
        {
            List<USBDeviceInfo> devices = new List<USBDeviceInfo>();

            ManagementObjectCollection collection;
            using (var searcher = new ManagementObjectSearcher(@"Select * From Win32_SerialPort"))

                collection = searcher.Get();

            foreach (var device in collection)
            {
                devices.Add(new USBDeviceInfo(
                    (string)device.GetPropertyValue("Name"),
                    (string)device.GetPropertyValue("DeviceID"),
                    (string)device.GetPropertyValue("PNPDeviceID"),
                    (string)device.GetPropertyValue("Description")
                ));
            }

            collection.Dispose();
            return devices;
        }

        public void UpdateRead(Boolean state)
        {
            updateread = state;
        }

        private void Write_CRC(byte[] data)
        {
            string str = "";
            byte crc = 0;
            Int32 i;
            for (i = 0; i < data.GetLength(0); i++)
            {
                crc += data[i];
                str += string.Format("{0:X2} ", data[i]);
            }
            data[data.GetLength(0) - 1] = (byte)(crc & 0x7F);
            str += string.Format("{0:X2} ", data[data.GetLength(0) - 1]);
            serialPort.Write(data, 0, i);

            debug.AppendText(str+"\n");
        }

        private bool readcmd(byte address, byte cmd, byte len, out byte[] arr)
        {
            string str = "";

            arr = new byte[len];

            bool valid = false;
            byte crc = address;
            crc += cmd;
            serialPort.Write(new byte[] { address }, 0, 1);
            str += string.Format("{0:X2} ", address);
            serialPort.Write(new byte[] { cmd }, 0, 1);
            str += string.Format("{0:X2} ", cmd);
            for (Int32 i = 0; i < len; i++)
            {
                arr[i] = (byte)serialPort.ReadByte();
                crc += arr[i];
                str += string.Format("{0:X2} ", arr[i]);
            }
            byte ccrc = (byte)serialPort.ReadByte();
            if ((crc & 0x7F) == ccrc)
                valid = true;
            else
            {
                valid = false;
            }
            str += string.Format("{0:X2} ", ccrc);

            if(updateread==true)
                debug.AppendText(str + "\n");

            return valid;
        }

        public bool IsOpen()
        {
            if(serialPort!=null)
                return serialPort.IsOpen;
            return false;
        }

        public void Close()
        {
            if (serialPort.IsOpen)
                serialPort.Close();
        }

        public bool Open(string comport,Int32 baudrate,System.Windows.Forms.TextBox textbox)
        {
            debug = textbox;

            if (comport == "AUTO")
            {
                //Close port if its already open
                if (serialPort != null)
                    if (serialPort.IsOpen)
                        serialPort.Close();

                //This code will autodetect USB Roboclaw boards
                var usbDevices = GetUSBDevices();   //Get all USB to Serial adapters
                foreach (var usbDevice in usbDevices)
                {
                    //Find just Atmel CDC Virtual Comports
                    if (usbDevice.PnpDeviceID.ToString().Substring(0, 21) == "USB\\VID_03EB&PID_2404")
                    {
                        //Check if this is a USB Roboclaw
                        comport = usbDevice.DeviceID;
                        baudrate = 2000000;
                    }
                }

                if (comport == "AUTO")
                    return false;
            }

            if (serialPort == null)
            {
                serialPort = new System.IO.Ports.SerialPort(comport, baudrate);
                if (serialPort == null)
                    return false;
            }
            else
            {
                if (serialPort.IsOpen)
                    serialPort.Close();
                serialPort.PortName = comport;
                serialPort.BaudRate = baudrate;
            }

            serialPort.Open();
            serialPort.ReadTimeout = 500;

            return true;
        }

        public void ST_M1Forward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 0, pwr, 0 });
        }

        public void ST_M2Forward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 4, pwr, 0 });
        }

        public void ST_M1Backward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 1, pwr, 0 });
        }

        public void ST_M2Backward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 5, pwr, 0 });
        }

        public void ST_M1Drive(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 6, pwr, 0 });
        }

        public void ST_M2Drive(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 7, pwr, 0 });
        }

        public void ST_MixedForward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 8, pwr, 0 });
        }

        public void ST_MixedBackward(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 9, pwr, 0 });
        }

        public void ST_MixedLeft(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 11, pwr, 0 });
        }

        public void ST_MixedRight(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 10, pwr, 0 });
        }

        public void ST_MixedDrive(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 12, pwr, 0 });
        }

        public void ST_MixedTurn(byte pwr)
        {
            Write_CRC(new byte[] { 0x80, 13, pwr, 0 });
        }

        public void ST_SetMinMainVoltage(byte set)
        {
            Write_CRC(new byte[] { 0x80, 2, set, 0 });
        }

        public void ST_SetMaxMainVoltage(byte set)
        {
            Write_CRC(new byte[] { 0x80, 3, set, 0 });
        }

        public bool GetM1Encoder(out Int32 enc,out byte status)
        {
            byte[] arr;
            if(readcmd(0x80, 16, 5, out arr)){
                enc = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            enc = 0;
            status = 0;
            return false;
        }

        public bool GetM2Encoder(out Int32 enc, out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 17, 5, out arr))
            {
                enc = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            enc = 0;
            status = 0;
            return false;
        }

        public bool GetM1Speed(out Int32 speed,out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 18, 5, out arr))
            {
                speed = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            speed = 0;
            status = 0;
            return false;
        }

        public bool GetM2Speed(out Int32 speed, out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 19, 5, out arr))
            {
                speed = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            speed = 0;
            status = 0;
            return false;
        }

        public void ResetEncoders()
        {
            Write_CRC(new byte[] { 0x80, 20, 0 });
        }

        public bool GetVersion(out string version)
        {
            serialPort.Write(new byte[] { 0x80, 21 }, 0, 2);
            try
            {
                version = serialPort.ReadLine();
                serialPort.ReadByte();
                serialPort.ReadByte();
                return true;
            }
            catch (TimeoutException)
            {
                version = "";
            }
            return false;
        }

        public bool GetMainVoltage(out double voltage)
        {
            voltage = 0;
            byte[] arr;
            if (readcmd(0x80, 24, 2, out arr))
            {
                voltage = (double)(arr[0] << 8 | arr[1])/10.0;
                return true;
            }
            return false;
        }

        public bool GetLogicVoltage(out double voltage)
        {
            voltage = 0;
            byte[] arr;
            if (readcmd(0x80, 25, 2, out arr))
            {
                voltage = (double)(arr[0] << 8 | arr[1])/10.0;
                return true;
            }
            return false;
        }

        public void ST_SetMaxLogicVoltage(byte set)
        {
            Write_CRC(new byte[] { 0x80, 26, set, 0 });
        }

        public void ST_SetMinLogicVolage(byte set)
        {
            Write_CRC(new byte[] { 0x80, 27, set, 0 });
        }

        public void SetM1VelocityConstants(double P, double I, double D, UInt32 qpps)
        {
            UInt32 p, i, d;
            p = (UInt32)(P * 65536.0);
            i = (UInt32)(I * 65536.0);
            d = (UInt32)(D * 65536.0);

            Write_CRC(new byte[] { 0x80, 28, (byte)(d >> 24), (byte)(d >> 16), (byte)(d >> 8), (byte)d,
                                          (byte)(p >> 24), (byte)(p >> 16), (byte)(p >> 8), (byte)p,
                                          (byte)(i >> 24), (byte)(i >> 16), (byte)(i >> 8), (byte)i,
                                          (byte)(qpps >> 24), (byte)(qpps >> 16), (byte)(qpps >> 8), (byte)qpps,
                                          0 });
        }

        public void SetM2VelocityConstants(double P, double I, double D, UInt32 qpps)
        {
            UInt32 p, i, d;
            p = (UInt32)(P * 65536.0);
            i = (UInt32)(I * 65536.0);
            d = (UInt32)(D * 65536.0);

            Write_CRC(new byte[] { 0x80, 29, (byte)(d >> 24), (byte)(d >> 16), (byte)(d >> 8), (byte)d,
                                          (byte)(p >> 24), (byte)(p >> 16), (byte)(p >> 8), (byte)p,
                                          (byte)(i >> 24), (byte)(i >> 16), (byte)(i >> 8), (byte)i,
                                          (byte)(qpps >> 24), (byte)(qpps >> 16), (byte)(qpps >> 8), (byte)qpps,
                                          0 });
        }

        public bool GetM1ISpeed(out Int32 speed, out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 30, 5, out arr))
            {
                speed = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            speed = 0;
            status = 0;
            return false;
        }

        public bool GetM2ISpeed(out Int32 speed, out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 31, 5, out arr))
            {
                speed = (Int32)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]);
                status = arr[4];
                return true;
            }
            speed = 0;
            status = 0;
            return false;
        }

        public void M1Duty(Int32 duty)
        {
            Write_CRC(new byte[] { 0x80, 32, (byte)(duty >> 8), (byte)duty, 0 });
        }

        public void M2Duty(Int32 duty)
        {
            Write_CRC(new byte[] { 0x80, 33, (byte)(duty >> 8), (byte)duty, 0 });
        }

        public void MixedDuty(Int16 duty1, Int16 duty2)
        {
            Write_CRC(new byte[] { 0x80, 34, (byte)(duty1 >> 8), (byte)duty1, (byte)(duty2 >> 8), (byte)duty2, 0 });
        }

        public void M1Speed(Int32 speed)
        {
            Write_CRC(new byte[] { 0x80, 35, (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed, 0 });
        }

        public void M2Speed(Int32 speed)
        {
            Write_CRC(new byte[] { 0x80, 36, (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed, 0 });
        }

        public void MixedSpeed(Int32 speed1, Int32 speed2)
        {
            Write_CRC(new byte[] { 0x80, 37, (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             0 });
        }

        public void M1SpeedAccel(UInt32 accel, Int32 speed)
        {
            Write_CRC(new byte[] { 0x80, 38, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             0 });
        }

        public void M2SpeedAccel(UInt32 accel, Int32 speed)
        {
            Write_CRC(new byte[] { 0x80, 39, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             0 });
        }

        public void MixedSpeedAccel(UInt32 accel, Int32 speed1, Int32 speed2)
        {
            Write_CRC(new byte[] { 0x80, 40, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             0 });
        }

        public void M1SpeedDistance(Int32 speed, UInt32 distance, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 41, (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(distance >> 24), (byte)(distance >> 16), (byte)(distance >> 8), (byte)distance,
                                             buffer, 0 });
        }

        public void M2SpeedDistance(Int32 speed, UInt32 distance, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 42, (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(distance >> 24), (byte)(distance >> 16), (byte)(distance >> 8), (byte)distance,
                                             buffer, 0 });
        }

        public void MixedSpeedDistance(Int32 speed1, UInt32 distance1, Int32 speed2, UInt32 distance2, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 43, (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(distance1 >> 24), (byte)(distance1 >> 16), (byte)(distance1 >> 8), (byte)distance1,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             (byte)(distance2 >> 24), (byte)(distance1 >> 16), (byte)(distance2 >> 8), (byte)distance2,
                                             buffer, 0 });
        }

        public void M1SpeedAccelDistance(Int32 accel, UInt32 speed, UInt32 distance, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 44, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(distance >> 24), (byte)(distance >> 16), (byte)(distance >> 8), (byte)distance,
                                             buffer, 0 });
        }

        public void M2SpeedAccelDistance(Int32 accel, UInt32 speed, UInt32 distance, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 45, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(distance >> 24), (byte)(distance >> 16), (byte)(distance >> 8), (byte)distance,
                                             buffer, 0 });
        }

        public void MixedSpeedAccelDistance(UInt32 accel, Int32 speed1, UInt32 distance1, Int32 speed2, UInt32 distance2, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 46, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(distance1 >> 24), (byte)(distance1 >> 16), (byte)(distance1 >> 8), (byte)distance1,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             (byte)(distance2 >> 24), (byte)(distance1 >> 16), (byte)(distance2 >> 8), (byte)distance2,
                                             buffer, 0 });
        }

        public bool GetBuffers(out byte buffer1, out byte buffer2)
        {
            byte[] arr;
            if (readcmd(0x80, 47, 2, out arr))
            {
                buffer1 = arr[0];
                buffer2 = arr[1];
                return true;
            }
            buffer1 = buffer2 = 0;
            return false;
        }

        public bool GetCurrents(out UInt16 current1, out UInt16 current2)
        {
            byte[] arr;
            if (readcmd(0x80, 49, 4, out arr))
            {
                current1 = (UInt16)(arr[0] << 8 | arr[1]);
                current2 = (UInt16)(arr[2] << 8 | arr[3]);
                return true;
            }
            current1 = current2 = 0;
            return false;
        }

        public void MixedSpeedAccel2(Int32 accel1, UInt32 speed1, Int32 accel2, UInt32 speed2)
        {
            Write_CRC(new byte[] { 0x80, 50, (byte)(accel1 >> 24), (byte)(accel1 >> 16), (byte)(accel1 >> 8), (byte)accel1,
                                             (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(accel2 >> 24), (byte)(accel2 >> 16), (byte)(accel2 >> 8), (byte)accel2,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             0 });
        }

        public void MixedSpeedAccelDistance2(Int32 accel1, UInt32 speed1, UInt32 distance1, Int32 accel2, UInt32 speed2, UInt32 distance2, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 51, (byte)(accel1 >> 24), (byte)(accel1 >> 16), (byte)(accel1 >> 8), (byte)accel1,
                                             (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(distance1 >> 24), (byte)(distance1 >> 16), (byte)(distance1 >> 8), (byte)distance1,
                                             (byte)(accel2 >> 24), (byte)(accel2 >> 16), (byte)(accel2 >> 8), (byte)accel2,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             (byte)(distance2 >> 24), (byte)(distance1 >> 16), (byte)(distance2 >> 8), (byte)distance2,
                                             buffer, 0 });
        }

        public void M1DutyAccel(Int16 duty, UInt16 accel)
        {
            Write_CRC(new byte[] { 0x80, 52, (byte)(duty >> 8), (byte)duty,
                                             (byte)(accel >> 8), (byte)accel,
                                             0 });
        }

        public void M2DutyAccel(Int16 duty, UInt16 accel)
        {
            Write_CRC(new byte[] { 0x80, 53, (byte)(duty >> 8), (byte)duty,
                                             (byte)(accel >> 8), (byte)accel,
                                             0 });
        }

        public void MixedDutyAccel(Int16 duty1, UInt16 accel1, Int16 duty2, UInt16 accel2)
        {
            Write_CRC(new byte[] { 0x80, 54, (byte)(duty1 >> 8), (byte)duty1,
                                             (byte)(accel1 >> 8), (byte)accel1,
                                             (byte)(duty2 >> 8), (byte)duty2,
                                             (byte)(accel2 >> 8), (byte)accel2,
                                             0 });
        }

        public bool GetM1VelocityConstants(out double p, out double i, out double d, out UInt32 qpps)
        {
            byte[] arr;
            if (readcmd(0x80, 55, 16, out arr))
            {
                p = (double)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]) / 65536;
                i = (double)(arr[4] << 24 | arr[5] << 16 | arr[6] << 8 | arr[7]) / 65536;
                d = (double)(arr[8] << 24 | arr[9] << 16 | arr[10] << 8 | arr[11]) / 65536;
                qpps = (UInt32)(arr[12] << 24 | arr[13] << 16 | arr[14] << 8 | arr[15]);
                return true;
            }
            p = i = d = qpps = 0;
            return false;
        }

        public bool GetM2VelocityConstants(out double p, out double i, out double d, out UInt32 qpps)
        {
            byte[] arr;
            if (readcmd(0x80, 56, 16, out arr))
            {
                p = (double)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]) / 65536;
                i = (double)(arr[4] << 24 | arr[5] << 16 | arr[6] << 8 | arr[7]) / 65536;
                d = (double)(arr[8] << 24 | arr[9] << 16 | arr[10] << 8 | arr[11]) / 65536;
                qpps = (UInt32)(arr[12] << 24 | arr[13] << 16 | arr[14] << 8 | arr[15]);
                return true;
            }
            p = i = d = qpps = 0;
            return false;
        }

        public void SetMainVoltageLimits(double Min, double Max)
        {
            UInt16 min = (UInt16)(Min * 10.0);
            UInt16 max = (UInt16)(Max * 10.0);
            Write_CRC(new byte[] { 0x80, 57, (byte)(min >> 8), (byte)min,
                                             (byte)(max >> 8), (byte)max,
                                             0 });
        }

        public void SetLogicVoltageLimits(double Min, double Max)
        {
            UInt16 min = (UInt16)(Min * 10.0);
            UInt16 max = (UInt16)(Max * 10.0);
            Write_CRC(new byte[] { 0x80, 58, (byte)(min >> 8), (byte)min,
                                             (byte)(max >> 8), (byte)max,
                                             0 });
        }

        public bool GetMainVoltageLimits(out double min, out double max)
        {
            byte[] arr;
            if (readcmd(0x80, 59, 4, out arr))
            {
                min = (double)(arr[0] << 8 | arr[1])/10.0;
                max = (double)(arr[2] << 8 | arr[3])/10.0;
                return true;
            }
            min = max = 0;
            return false;
        }

        public bool GetLogicVoltageLimits(out double min, out double max)
        {
            byte[] arr;
            if (readcmd(0x80, 60, 4, out arr))
            {
                min = (double)(arr[0] << 8 | arr[1])/10.0;
                max = (double)(arr[2] << 8 | arr[3])/10.0;
                return true;
            }
            min = max = 0;
            return false;
        }

        public void SetM1PositionConstants(double P, double I, double D, UInt32 imax, UInt32 deadzone, Int32 minlimit, Int32 maxlimit)
        {
            UInt32 p, i, d;
            p = (UInt32)(P * 1024.0);
            i = (UInt32)(I * 1024.0);
            d = (UInt32)(D * 1024.0);

            Write_CRC(new byte[] { 0x80, 61, (byte)(d >> 24), (byte)(d >> 16), (byte)(d >> 8), (byte)d,
                                          (byte)(p >> 24), (byte)(p >> 16), (byte)(p >> 8), (byte)p,
                                          (byte)(i >> 24), (byte)(i >> 16), (byte)(i >> 8), (byte)i,
                                          (byte)(imax >> 24), (byte)(imax >> 16), (byte)(imax >> 8), (byte)imax,
                                          (byte)(deadzone >> 24), (byte)(deadzone >> 16), (byte)(deadzone >> 8), (byte)deadzone,
                                          (byte)(minlimit >> 24), (byte)(minlimit >> 16), (byte)(minlimit >> 8), (byte)minlimit,
                                          (byte)(maxlimit >> 24), (byte)(maxlimit >> 16), (byte)(maxlimit >> 8), (byte)maxlimit,
                                          0 });
        }

        public void SetM2PositionConstants(double P, double I, double D, UInt32 imax, UInt32 deadzone, Int32 minlimit, Int32 maxlimit)
        {
            UInt32 p, i, d;
            p = (UInt32)(P * 1024.0);
            i = (UInt32)(I * 1024.0);
            d = (UInt32)(D * 1024.0);

            Write_CRC(new byte[] { 0x80, 62, (byte)(d >> 24), (byte)(d >> 16), (byte)(d >> 8), (byte)d,
                                          (byte)(p >> 24), (byte)(p >> 16), (byte)(p >> 8), (byte)p,
                                          (byte)(i >> 24), (byte)(i >> 16), (byte)(i >> 8), (byte)i,
                                          (byte)(imax >> 24), (byte)(imax >> 16), (byte)(imax >> 8), (byte)imax,
                                          (byte)(deadzone >> 24), (byte)(deadzone >> 16), (byte)(deadzone >> 8), (byte)deadzone,
                                          (byte)(minlimit >> 24), (byte)(minlimit >> 16), (byte)(minlimit >> 8), (byte)minlimit,
                                          (byte)(maxlimit >> 24), (byte)(maxlimit >> 16), (byte)(maxlimit >> 8), (byte)maxlimit,
                                          0 });
        }

        public bool GetM1PositionConstants(out double p, out double i, out double d, out UInt32 imax, out UInt32 deadzone, out Int32 minlimit, out Int32 maxlimit)
        {
            byte[] arr;
            if (readcmd(0x80, 63, 28, out arr))
            {
                p = (double)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]) / 1024;
                i = (double)(arr[4] << 24 | arr[5] << 16 | arr[6] << 8 | arr[7]) / 1024;
                d = (double)(arr[8] << 24 | arr[9] << 16 | arr[10] << 8 | arr[11]) / 1024;
                imax = (UInt32)(arr[12] << 24 | arr[13] << 16 | arr[14] << 8 | arr[15]);
                deadzone = (UInt32)(arr[16] << 24 | arr[17] << 16 | arr[18] << 8 | arr[19]);
                minlimit = (Int32)(arr[20] << 24 | arr[21] << 16 | arr[22] << 8 | arr[23]);
                maxlimit = (Int32)(arr[24] << 24 | arr[25] << 16 | arr[26] << 8 | arr[27]);
                return true;
            }
            p = i = d = imax = deadzone = 0;
            minlimit = maxlimit = 0;
            return false;
        }

        public bool GetM2PositionConstants(out double p, out double i, out double d, out UInt32 imax, out UInt32 deadzone, out Int32 minlimit, out Int32 maxlimit)
        {
            byte[] arr;
            if (readcmd(0x80, 64, 28, out arr))
            {
                p= (double)(arr[0] << 24 | arr[1] << 16 | arr[2] << 8 | arr[3]) / 1024;
                i = (double)(arr[4] << 24 | arr[5] << 16 | arr[6] << 8 | arr[7]) / 1024;
                d = (double)(arr[8] << 24 | arr[9] << 16 | arr[10] << 8 | arr[11]) / 1024;
                imax = (UInt32)(arr[12] << 24 | arr[13] << 16 | arr[14] << 8 | arr[15]);
                deadzone = (UInt32)(arr[16] << 24 | arr[17] << 16 | arr[18] << 8 | arr[19]);
                minlimit = (Int32)(arr[20] << 24 | arr[21] << 16 | arr[22] << 8 | arr[23]);
                maxlimit = (Int32)(arr[24] << 24 | arr[25] << 16 | arr[26] << 8 | arr[27]);
                return true;
            }
            p = i = d = imax = deadzone = 0;
            minlimit = maxlimit = 0;
            return false;
        }

        public void M1SpeedAccelDeccelPosition(UInt32 accel, UInt32 speed, UInt32 deccel, Int32 position, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 65, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(deccel >> 24), (byte)(deccel >> 16), (byte)(deccel >> 8), (byte)deccel,
                                             (byte)(position >> 24), (byte)(position >> 16), (byte)(position >> 8), (byte)position,
                                             buffer, 0 });
        }

        public void M2SpeedAccelDeccelPosition(UInt32 accel, UInt32 speed, UInt32 deccel, Int32 position, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 66, (byte)(accel >> 24), (byte)(accel >> 16), (byte)(accel >> 8), (byte)accel,
                                             (byte)(speed >> 24), (byte)(speed >> 16), (byte)(speed >> 8), (byte)speed,
                                             (byte)(deccel >> 24), (byte)(deccel >> 16), (byte)(deccel >> 8), (byte)deccel,
                                             (byte)(position >> 24), (byte)(position >> 16), (byte)(position >> 8), (byte)position,
                                             buffer, 0 });
        }

        public void MixedSpeedAccelDeccelPosition(UInt32 accel1, UInt32 speed1, UInt32 deccel1, Int32 position1, UInt32 accel2, UInt32 speed2, UInt32 deccel2, Int32 position2, byte buffer)
        {
            Write_CRC(new byte[] { 0x80, 67, (byte)(accel1 >> 24), (byte)(accel1 >> 16), (byte)(accel1 >> 8), (byte)accel1,
                                             (byte)(speed1 >> 24), (byte)(speed1 >> 16), (byte)(speed1 >> 8), (byte)speed1,
                                             (byte)(deccel1 >> 24), (byte)(deccel1 >> 16), (byte)(deccel1 >> 8), (byte)deccel1,
                                             (byte)(position1 >> 24), (byte)(position1 >> 16), (byte)(position1 >> 8), (byte)position1,
                                             (byte)(accel2 >> 24), (byte)(accel2 >> 16), (byte)(accel2 >> 8), (byte)accel2,
                                             (byte)(speed2 >> 24), (byte)(speed2 >> 16), (byte)(speed2 >> 8), (byte)speed2,
                                             (byte)(deccel2 >> 24), (byte)(deccel2 >> 16), (byte)(deccel2 >> 8), (byte)deccel2,
                                             (byte)(position2 >> 24), (byte)(position2 >> 16), (byte)(position2 >> 8), (byte)position2,
                                             buffer, 0 });
        }

        public bool GetTemperature(out double temperature)
        {
            byte[] arr;
            if (readcmd(0x80, 82, 2, out arr))
            {
                temperature = (double)(arr[0] << 8 | arr[1])/10.0;
                return true;
            }
            temperature = 0;
            return false;
        }

        public bool GetStatus(out byte status)
        {
            byte[] arr;
            if (readcmd(0x80, 90, 1, out arr))
            {
                status = arr[0];
                return true;
            }
            status = 0;
            return false;
        }

        public bool GetEncoderModes(out byte m1mode, out byte m2mode)
        {
            byte[] arr;
            if (readcmd(0x80, 91, 2, out arr))
            {
                m1mode = arr[0];
                m2mode = arr[1];
                return true;
            }
            m1mode = m2mode = 0;
            return false;
        }

        public void SetEncoder1Mode(byte mode)
        {
            Write_CRC(new byte[] { 0x80, 92, mode, 0 });
        }

        public void SetEncoder2Mode(byte mode)
        {
            Write_CRC(new byte[] { 0x80, 93, mode, 0 });
        }

        public void WriteNVM()
        {
            Write_CRC(new byte[] { 0x80, 94, 0 });
        }

        public void SetConfig(UInt16 config)
        {
            Write_CRC(new byte[] { 0x80, 98, (byte)(config>>8),(byte)config, 0 });
        }

        public bool GetConfig(out UInt16 config)
        {
            byte[] arr;
            if (readcmd(0x80, 99, 2, out arr))
            {
                config = (UInt16)(arr[0] << 8 | arr[1]);
                return true;
            }
            config = 0;
            return false;
        }
    }
}
